﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;

namespace VA.PPMS.CRM.Plugins.Agreement
{
    public class AgreementUpdate : IPlugin
    {
        private const string PluginName = "ProviderUpdate";
        private const string MessageCreate = "CREATE";
        private const string MessageUpdate = "UPDATE";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            ITracingService tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity provider = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (provider.LogicalName != "ppms_provideragreement")
                    return;

                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Get related entities to deactivate
                    tracingService.Trace("Retrieve Provider and related entities");
                    Entity entity = GetAgreement(service, provider.Id, tracingService);
                    if (entity == null)
                    {
                        tracingService.Trace("Failed to retrieve provider related entities");
                        return;
                    }
                    tracingService.Trace("Provider Retrieved");

                    List<SetStateRequest> requests = new List<SetStateRequest>();

                    // handle event based on message type
                    tracingService.Trace("Check status");
                    var isActive = PpmsHelper.IsActive(entity);
                    int state = 0;
                    int statuscode = 0;

                    if (isActive)
                    {
                        tracingService.Trace("Provider in Active Status");
                        state = (int)PpmsHelper.AccountState.Active;
                        statuscode = (int)PpmsHelper.Account_StatusCode.Active;
                    }
                    else
                    {
                        tracingService.Trace("Provider in Inactive Status");
                        state = (int)PpmsHelper.AccountState.Inactive;
                        statuscode = (int)PpmsHelper.Account_StatusCode.Inactive;
                    }

                    tracingService.Trace("Adding Set State Requests to the related entities");
                    requests.Add(PpmsHelper.GetStateRequest(entity, state, statuscode));

                    // provider service
                    requests.AddRange(GetEntityStateRequests(entity, "ppms_provideragreement_ppms_providerservice_ProviderAgreement", state, statuscode));

                    if (requests != null)
                    {
                        tracingService.Trace("Exectuting Set State Requests");
                        foreach (var request in requests)
                        {
                            service.Execute(request);
                        }
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }

        private IEnumerable<SetStateRequest> GetEntityStateRequests(Entity entity, string schemaName, int state, int statuscode)
        {
            List<SetStateRequest> requests = new List<SetStateRequest>();

            var children = entity.RelatedEntities[new Relationship(schemaName)];
            if (children != null)
            {
                foreach (var child in children.Entities)
                {
                    requests.Add(PpmsHelper.GetStateRequest(child, state, statuscode));
                }
            }

            return requests;
        }

        private Entity GetAgreement(IOrganizationService service, Guid agreementId, ITracingService tracingService)
        {
            var request = new RetrieveRequest();

            request.Target = new EntityReference("ppms_provideragreement", agreementId);
            request.ColumnSet = new ColumnSet(new string[] { "ppms_provideragreementid", "name", "statecode" });
            request.RelatedEntitiesQuery = new RelationshipQueryCollection();

            //Retrieve related entities
            request.RelatedEntitiesQuery.Add(new Relationship("ppms_provideragreement_ppms_providerservice_ProviderAgreement"),
                new QueryExpression("ppms_providerservice")
                {
                    ColumnSet = new ColumnSet("ppms_providerserviceid", "ppms_name")
                }
            );

            //Get response
            tracingService.Trace("Excecuting request to retrieve Agreement and relationships");
            var response = (RetrieveResponse)service.Execute(request);
            if (response != null)
                return response.Entity;

            return null;
        }
    }
}
